//========================================================================
//	Acquire.cpp
//	Copyright 1995 - 1996 Metrowerks Corporation, All Rights Reserved.
//========================================================================	

#include <Debug.h>

#include "Acquire.h"

#ifndef LOG
	#define LOG(x)
#endif
#ifndef PRINT
	#define PRINT(x)
#endif


sem_id Acquire::sStructSem;
Acquire::SemStruct * Acquire::sSemStructs;


//	Call this before the first Acquire object becomes active
long
Acquire::OpenAcquire()
{
	sStructSem = create_sem(1, "Acquire Semaphore");
	if (sStructSem < B_NO_ERROR)
	{
		long ret = sStructSem;
		sStructSem = 0;
		return ret;
	}
	return B_NO_ERROR;
}


//	Call this when no more Acquire objects are or will be active
void
Acquire::CloseAcquire()
{
	if (!sStructSem)
		return;
	acquire_sem(sStructSem);
	SemStruct * ptr = sSemStructs;
	sSemStructs = NULL;
	while (ptr)
	{
		ASSERT(!ptr->fAcquired);
#if defined(DEBUG)
		if (ptr->fAcquired)
		{
			LOG(ptr->fSem);
			LOG(ptr->fThread);
		}
#endif
		SemStruct * del = ptr;
		ptr = ptr->fNext;
		delete del;
	}
	delete_sem(sStructSem);
}


Acquire::Acquire(
	sem_id sem,
	const char *)	//	Ignore text for now
{
	fHead = NULL;
	fNext = NULL;
	ASSERT(sStructSem != 0);

	thread_id thread = find_thread(NULL);
	if (thread < B_NO_ERROR)
	{
		PRINT(("find_thread Error %lx", thread));
		return;
	}

	//	Find if we already have it
	acquire_sem(sStructSem);
	volatile SemStruct * ptr = sSemStructs;
	while (ptr != NULL)
	{
		if (ptr->fSem == sem)
		{
			fHead = ptr;
			if (ptr->fThread == thread)	//	we're already there? good!
			{
				fNext = ptr->fAcquired;
				ptr->fAcquired = this;
				release_sem(sStructSem);
				return;
			}
			//	So we know may have to wait
			break;
		}
		ptr = ptr->fNext;
	}
	release_sem(sStructSem);

	//	Acquire it
	//	If this looks like a race condition, think again.
	//	Hint: it is benign.
	long err = acquire_sem(sem);
	ASSERT(err == B_NO_ERROR);
	if (err != B_NO_ERROR)
	{
		PRINT(("Error %08x", err));
		fHead = NULL;
		return;
	}

	//	Make note of us
	acquire_sem(sStructSem);
	ptr = sSemStructs;
	while (ptr != NULL)
	{
		if (ptr->fSem == sem)
			break;
		ptr = ptr->fNext;
	}
	if (!ptr)
	{
		ptr = new SemStruct;
		ptr->fAcquired = NULL;
		ptr->fSem = sem;
		ptr->fNext = sSemStructs;
		sSemStructs = ptr;
	}
	fHead = ptr;
	ptr->fThread = thread;
	fNext = ptr->fAcquired;
	ptr->fAcquired = this;
	release_sem(sStructSem);
}


Acquire::~Acquire()
{
	if (!fHead)
	{
		PRINT(("fHead is NULL"));
		return;
	}

	//	Find ourselves
	acquire_sem(sStructSem);
	ASSERT(fHead->fAcquired == this);	//	Should come in reverse order
	ASSERT(find_thread(NULL) == fHead->fThread);
#if DEBUG
	if (fHead->fAcquired != this)
	{
		Acquire * ptr = fHead->fAcquired;
		while (ptr != NULL)
		{
			FPRINTF((stderr, "Non-matching 0x%08lx", ptr));
			ptr = ptr->fNext;
			if (ptr == this)
			{
				fHead->fAcquired = this;
				break;
			}
		}
	}
#endif

	//	Unlink us
	fHead->fAcquired = fHead->fAcquired->fNext;

	//	If we were last, release the semaphore
	if (!fHead->fAcquired)
	{
		fHead->fThread = -1;
		release_sem(fHead->fSem);
	}

	release_sem(sStructSem);
}
